# chillerlan/php-traits

A collection of (more or less) useful traits for PHP7+
  
[![version][packagist-badge]][packagist]
[![license][license-badge]][license]
[![Travis][travis-badge]][travis]
[![Coverage][coverage-badge]][coverage]
[![Scrunitizer][scrutinizer-badge]][scrutinizer]
[![Packagist downloads][downloads-badge]][downloads]
[![PayPal donate][donate-badge]][donate]

[packagist-badge]: https://img.shields.io/packagist/v/chillerlan/php-traits.svg?style=flat-square
[packagist]: https://packagist.org/packages/chillerlan/php-traits
[license-badge]: https://img.shields.io/github/license/chillerlan/php-traits.svg?style=flat-square
[license]: https://github.com/chillerlan/php-traits/blob/master/LICENSE
[travis-badge]: https://img.shields.io/travis/chillerlan/php-traits.svg?style=flat-square
[travis]: https://travis-ci.org/chillerlan/php-traits
[coverage-badge]: https://img.shields.io/codecov/c/github/chillerlan/php-traits.svg?style=flat-square
[coverage]: https://codecov.io/github/chillerlan/php-traits
[scrutinizer-badge]: https://img.shields.io/scrutinizer/g/chillerlan/php-traits.svg?style=flat-square
[scrutinizer]: https://scrutinizer-ci.com/g/chillerlan/php-traits
[downloads-badge]: https://img.shields.io/packagist/dt/chillerlan/php-traits.svg?style=flat-square
[downloads]: https://packagist.org/packages/chillerlan/php-traits/stats
[donate-badge]: https://img.shields.io/badge/donate-paypal-ff33aa.svg?style=flat-square
[donate]: https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=WLYUNAT9ZTJZ4

## Features
- `ClassLoader` - invokes objects of a given class and interface/type with an arbitrary count of constructor arguments
- `Container` - provides immutable properties with magic getter & setter and some fancy, implements `ContainerInterface`
- `Magic` - turns methods into magic properties
- `Enumerable` - provides some of [prototype's enumerable methods](http://api.prototypejs.org/language/Enumerable/), implements `EnumerableInterface`
- `Env` - loads contents from a `.env` file into the environment (similar to [vlucas/phpdotenv](https://github.com/vlucas/phpdotenv))
  - `DotEnv` - a standalone `Env` class
- `ArrayHelpers`
  - `ByteArray` - useful for byte/bit-flipping purposes, extends [`SplFixedArray`](http://php.net/manual/class.splfixedarray.php)
  - `ByteArrayDispenser` - creates `ByteArray` from several data types (hex, base64, binary, json etc.)
  - `DotArray` - adds dot key notation functionality
  - `SearchableArray` - deep search arrays using [`RecursiveIteratorIterator`](http://php.net/manual/class.recursiveiteratoriterator.php)
- `Crypto` (requires the [Sodium extension](http://php.net/manual/book.sodium.php))
  - `CryptoBoxInterface`: `Box`, `SecretBox`, `SealedBox`, `SignedMessage`
  - `CryptoKeyInterface`: `BoxKeypair`, `SignKeypair`
  - `CryptoTrait` provides shorthand methods for the above
  - `MemzeroDestructorTrait` - performs `sodium_memzero` on object variables when `__destruct()` is being called
- `Interfaces`
  - `ArrayAccessTrait` - implements [`ArrayAccess`](http://php.net/manual/class.arrayaccess.php)
  - `IteratorTrait` - implements [`Iterator`](http://php.net/manual/class.iterator.php)
  - `SerializableTrait` - implements [`Serializable`](http://php.net/manual/class.serializable.php)
- `SPL`
  - `CountableTrait` - implements [`Countable`](http://php.net/manual/class.countable.php)
  - `SeekableIteratorTrait` - implements [`SeekableIterator`](http://php.net/manual/class.seekableiterator.php)
 
## Documentation

### Installation
**requires [composer](https://getcomposer.org)**

*composer.json* (note: replace `dev-master` with a version boundary)
```json
{
	"require": {
		"php": ">=7.0.3",
		"chillerlan/php-traits": "dev-master"
	}
}
```

#### Manual installation
Download the desired version of the package from [master](https://github.com/chillerlan/php-traits/archive/master.zip) or 
[release](https://github.com/chillerlan/php-traits/releases) and extract the contents to your project folder.  After that:
- run `composer install` to install the required dependencies and generate `/vendor/autoload.php`.
- if you use a custom autoloader, point the namespace `chillerlan\Traits` to the folder `src` of the package 

Profit!

### Usage

#### `ClassLoader`
Simple usage:
```php
class MyClass{
	use ClassLoader;
	
	protected function doStuff(string $class){
		$obj = $this->loadClass(__NAMESPACE__.'\\Whatever\\'.$class);
		
		// do stuff
	}
}
```

Let's assume we have several classes that implement the same interface, but their constructors have different parameter counts, like so:
```php
class SomeClass implements MyInterface{
	public funtion __construct($param_foo){}
}

class OtherClass implements MyInterface{
	public funtion __construct($param_foo, $param_bar){}
}
```

Initialize an object based on a selction

```php
class MyClass{
	use ClassLoader;
	
	protected $classes = [
		'foo' => SomeClass::class, 
		'bar' => OtherClass::class
	];
	
	protected funtion initInterface(string $whatever, $foo, $bar = null):MyInterface{
	
		foreach($this->classes as $what => $class){
			if($whatever === $what){
				return $this->loadClass($class, MyInterface::class, $foo, $bar);
			}
		}
	
	}
}
```


#### `Container`
```php
class MyContainer implements ContainerInterface{
	use Container;

	protected $foo;
	protected $bar;
}
```

```php
// use it just like a \stdClass
$container = new MyContainer;
$container->foo = 'what';
$container->bar = 'foo';

// which is equivalent to 
$container = new MyContainer(['bar' => 'foo', 'foo' => 'what']);
// ...or try
$container->__fromJSON('{"foo": "what", "bar": "foo"}');


// fetch all properties as array
$container->__toArray(); // -> ['foo' => 'what', 'bar' => 'foo']
// or JSON
$container->__toJSON(); // -> {"foo": "what", "bar": "foo"}

//non-existing properties will be ignored:
$container->nope = 'what';

var_dump($container->nope); // -> null
```


#### `Magic`
`Magic` allows to access internal methods like as properties.
```php
class MyMagicContainer{
	use Magic;

	protected $foo;

	protected function magic_get_foo(){
		// do whatever...
		
		return 'foo: '.$this->foo;
	}

	protected function magic_set_foo($value){
		// do stuff with $value
		// ...
		
		$this->foo = $value.'bar';
	}
}
```

```php
$magic = new MyMagicContainer;

$magic->foo = 'foo';

var_dump($magic->foo); // -> foo: foobar

```

#### `Enumerable`
```php
class MyEnumerableContainer implements EnumerableInterface{
	use Enumerable;

	public function __construct(array $data){
		$this->array = $data;
	}
}
```

```php
$enum = new MyEnumerableContainer($data);

$enum
	->__each(function($value, $index){
		// do stuff
		
		$this->array[$index] = $stuff;
	})
	->__reverse()
	->__to_array()
;

$arr = $enum->__map(function($value, $index){
	// do stuff
	
	return $stuff;
});

$enum;

```

#### `Env`
```
# example .env
FOO=bar
BAR=foo
WHAT=${BAR}-${FOO}
```

```php
class MyClass{
	use Env;
	
	protected $foo;
	
	public function __construct(){
		// load and overwrite existing vars, require var "WHAT"
		$this->__loadEnv(__DIR__.'/../config', '.env', ['WHAT']);
		
		// will not overwrite
		$this->__addEnv(__DIR__.'/../config', '.env', false, ['WHAT']); 
		
		$this->foo = $_ENV['WHAT']; // -> foo-bar
		// or
		$this->foo = $this->__getEnv('WHAT');
	}
}
```

```php
$env = new DotEnv(__DIR__.'/../config', '.env');
$env->load(['foo']); // foo is required

$foo = $env->get('FOO'); // -> bar

$foo = $env->set('foo', 'whatever');
$foo = $env->get('FOO'); // -> whatever
```
